import serial
import tkinter as tk
from tkinter import ttk, simpledialog, messagebox

class GrocerySystem:
    def __init__(self, root):
        # Initialize the main window
        self.root = root
        self.root.title("Grocery System")

        # Initialize a dictionary to store cart items
        self.items = {}

        # Initialize customer information
        self.name = ""
        self.phone = ""

        # Initialize the inventory with sample items
        self.inventory = {
            "Dal": {"gram_price": 3.99, "kg_price": 39.99},
            "Rice": {"gram_price": 2.49, "kg_price": 24.99},
            "Wheat": {"gram_price": 1.99, "kg_price": 19.99},
        }

        # Configure ttk style
        self.style = ttk.Style()
        self.style.configure("TButton", font=("Helvetica", 12))

        # Create a frame for the left side
        left_frame = ttk.Frame(root, padding=20)
        left_frame.grid(row=0, column=0, padx=20, pady=20, sticky="nsew")

        # Create a frame for the right side
        right_frame = ttk.Frame(root, padding=20)
        right_frame.grid(row=0, column=1, padx=20, pady=20, sticky="nsew")

        # Create labels and widgets for customer info
        ttk.Label(left_frame, text="Name:").grid(row=0, column=0, sticky="w")
        self.name_var = tk.StringVar()
        self.name_entry = ttk.Entry(left_frame, textvariable=self.name_var)
        self.name_entry.grid(row=1, column=0)

        ttk.Label(left_frame, text="Phone:").grid(row=2, column=0, sticky="w")
        self.phone_var = tk.StringVar()
        self.phone_entry = ttk.Entry(left_frame, textvariable=self.phone_var)
        self.phone_entry.grid(row=3, column=0)

        # Create labels and widgets for the left frame
        ttk.Label(left_frame, text="Select Item:").grid(row=4, column=0, sticky="w")
        self.item_var = tk.StringVar()

        # Remove "Amount" from item choices
        item_choices = list(self.inventory.keys()) + ["Unknown Item"]
        self.item_combobox = ttk.Combobox(left_frame, textvariable=self.item_var, values=item_choices)
        self.item_combobox.grid(row=5, column=0)

        ttk.Label(left_frame, text="Enter Quantity:").grid(row=6, column=0, sticky="w")
        self.quantity_var = tk.DoubleVar()
        self.quantity_entry = ttk.Entry(left_frame, textvariable=self.quantity_var)
        self.quantity_entry.grid(row=7, column=0)

        ttk.Label(left_frame, text="Select Unit:").grid(row=8, column=0, sticky="w")
        self.unit_var = tk.StringVar()

        # Include "gram," "kg," and "amt" in the unit combobox
        unit_choices = ["gram", "kg", "amt"]
        self.unit_var.set("gram")  # Default unit is gram
        self.unit_combobox = ttk.Combobox(left_frame, textvariable=self.unit_var, values=unit_choices)
        self.unit_combobox.grid(row=9, column=0)

        self.add_button = ttk.Button(left_frame, text="Add Item", command=self.add_item)
        self.add_button.grid(row=10, column=0, pady=10)

        # Create labels and widgets for the right frame
        ttk.Label(right_frame, text="Cart:").grid(row=0, column=0, sticky="w")
        self.cart_listbox = tk.Listbox(right_frame, width=40, height=15, font=("Helvetica", 12))
        self.cart_listbox.grid(row=1, column=0, rowspan=4)

        self.delete_button = ttk.Button(right_frame, text="Delete Item", command=self.delete_item)
        self.delete_button.grid(row=5, column=0, pady=10)

        self.total_button = ttk.Button(right_frame, text="Total Amount", command=self.calculate_total)
        self.total_button.grid(row=6, column=0)

        self.send_button = ttk.Button(right_frame, text="Send", command=self.send_cart)
        self.send_button.grid(row=7, column=0)

        self.total_label = ttk.Label(right_frame, text="")
        self.total_label.grid(row=8, column=0)

        # Initialize serial communication with Arduino (Change 'COM4' to the appropriate port)
        self.arduino = serial.Serial('COM4', 9600)

    def add_item(self):
        # Get selected item, quantity, and unit
        item = self.item_var.get()
        quantity = self.quantity_var.get()
       
        # Get selected item, quantity, and unit
        item = self.item_var.get()
        quantity = self.quantity_var.get()
        unit = self.unit_var.get()

        # Check for valid input
        if not item or quantity <= 0:
            messagebox.showerror("Error", "Please select an item, enter a valid quantity, and choose a unit")
            return

        # Check if the item is valid
        if item != "Unknown Item" and item not in self.inventory:
            messagebox.showerror("Error", "Invalid item selected")
            return

        if item == "Unknown Item":
            # Ask for a custom item name
            custom_item_name = simpledialog.askstring("Custom Item", "Enter the item name:")
            if not custom_item_name:
                return  # User canceled the input

            # Ask for the custom item price from the user
            custom_item_price = simpledialog.askfloat("Custom Item", f"Enter the price for {custom_item_name} (₹):")
            if custom_item_price is None:
                return  # User canceled the input

            # Calculate the total price
            total_price = custom_item_price * quantity

            # Add the custom item to the cart
            self.items[custom_item_name] = {"quantity": quantity, "unit": "amt", "price": total_price}
            item = custom_item_name  # Use the custom item name

        else:
            # Calculate the total price based on unit
            if unit == "gram":
                price_per_unit = self.inventory[item]["gram_price"]
            elif unit == "kg":
                price_per_unit = self.inventory[item]["kg_price"]
            elif unit == "amt":
                price_per_unit = 1  # Price per custom item (1 unit)

            total_price = price_per_unit * quantity

            # Add the item to the cart
            self.items[item] = {"quantity": quantity, "unit": unit, "price": total_price}

        # Update the cart listbox with the added item
        self.cart_listbox.insert(tk.END, f"{item} - {quantity} {unit}(s): ₹{total_price:.2f}")

        # Clear input fields
        self.item_var.set("")
        self.quantity_var.set(0)

    def delete_item(self):
        # Get the selected item from the cart listbox
        selected_item_index = self.cart_listbox.curselection()
        if selected_item_index:
            selected_item_index = selected_item_index[0]
            item = self.cart_listbox.get(selected_item_index)
            item = item.split(" - ")[0]

            # Handle "Unknown Item" separately
            if item == "Unknown Item":
                unknown_item = simpledialog.askstring("Delete Item", "Enter the custom item name to delete:")
                if unknown_item in self.items:
                    del self.items[unknown_item]
                else:
                    messagebox.showerror("Error", f"Item '{unknown_item}' not found in the cart")
            else:
                del self.items[item]

            # Remove the item from the cart listbox
            self.cart_listbox.delete(selected_item_index)

    def calculate_total(self):
        # Calculate the total price of all items in the cart
        total = sum(item["price"] for item in self.items.values())

        # Update the total label with the calculated total
        self.total_label.config(text=f"Total Amount: ₹{total:.2f}")

    def send_cart(self):
        # Get customer name and phone number
        self.name = self.name_var.get()
        self.phone = self.phone_var.get()

        # Check if customer name and phone number are provided
        if not self.name or not self.phone:
            messagebox.showerror("Error", "Please enter name and phone number.")
            return

        # Calculate and display the total amount
        total = sum(item["price"] for item in self.items.values())

        # Prepare a message to send to Arduino
        cart_message = f"Name: {self.name}\nPhone: {self.phone}\nCart Items:\n"
        for item, data in self.items.items():
            cart_message += f"{item} - {data['quantity']} {data['unit']}(s): ₹{data['price']:.2f}\n"
        cart_message += f"Total Amount: ₹{total:.2f}\n"

        # Send the cart details to Arduino via serial communication
        self.arduino.write(cart_message.encode())

        # Clear the cart and customer information
        self.items = {}
        self.cart_listbox.delete(0, tk.END)
        self.total_label.config(text="")
        self.name_var.set("")
        self.phone_var.set("")

if __name__ == "__main__":
    root = tk.Tk()
    app = GrocerySystem(root)
    root.mainloop()
